/*******************************************************************************
 * Copyright () 2001 Ian Soanes (ians@lineo.com), All rights reserved
 *
 * Development of this (watchdog) module was sponsored by Alcatel, Strasbourg
 * as part of general debugging enhancements to RTAI.
 *
 *******************************************************************************
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA.
 *
 *******************************************************************************
 *
 *                           RTAI Watchdog Module
 *                           --------------------
 *
 * Module to provide various watchdog protection services to RTAI thereby
 * protecting it (and the host Linux OS) against programming errors in RTAI
 * applications.
 *
 * Services provided...
 *
 * 1. Detection of RT tasks that exceed their alloted time period. This will
 *    detect tasks that have gone into infinite loops or are regularly
 *    overunning. Normally such tasks would prevent other tasks (and Linux)
 *    from being scheduled and possibly lock and/or crash the system.
 *
 * 2. The ability to adjust important parameters when inserting the watchdog
 *    module and from other RT modules via a simple API.
 *
 * 3. Configurable policy to use on bad tasks. Currently available policies
 *    are...
 *
 *    o Do nothing, other than log some messages and keep a record of the bad
 *      task. In reality you will probably never get the chance to see these
 *      messages if the task is locking out the Linux task. This policy is not
 *      usually recommended.
 *
 *    o Resynchronise the task's frame time and nothing more. This is good for
 *      tasks that occasionally overrun. Doing this should prevent the system
 *      from locking up and crashing as the scheduler tries to catch up with the
 *      missed deadlines. The maximum (if any) number of times to resynchronise
 *      a task before permanently suspending it is configurable.
 *
 *    o Debug policy, this is a special case of the above resync policy. It is
 *      recommended when step and trace debugging RT tasks that use oneshot RT
 *      timer mode. (See README.WATCHDOG for full details)
 *
 *    o Stretch (increase) the period of the offending task until it no longer
 *      overruns. The percentage increment (of the original period) is
 *      configurable, as is the maximum (if any) number of times to increase
 *      the period before permanently suspending the task. When a task's period
 *      is increased in this way the scheduler is asked to resynchronise the
 *      task's frame time in order to prevent the system locking up and crashing
 *      as it tries to catch up with the missed deadlines. This policy could be
 *      a useful development aid if you are not sure what period to use for a
 *      task.
 *
 *    o Slip the offending task by forcibly suspending it for a percentage of
 *      its period. The percentage slip is configurable, as is the maximum (if
 *      any) number of times to slip the task before it is permanently
 *      suspended. By slipping the task, other tasks (including Linux) are
 *      given the oppurtunity to run and the system doesn't lock up.
 *
 *    o Suspend the offending task so that it no longer poses any threat to
 *      the system. The task will still be known to the scheduler so it could
 *      possibly be resumed sometime later.
 *
 *    o Kill the offending task and remove all trace of it.
 *
 * 4. A safety limit that will suspend any task that overruns excessively. The
 *    definition of 'excessive' is configurable and can also be disabled. This
 *    is designed to deal with infinite loops no matter what the current policy.
 *    The safety limit needs to be set sufficiently high so that it doesn't
 *    interfere with the prevailing watchdog policy. This limit is automatically
 *    disabled when the policy is set to 'Debug' in order not to suspend RT
 *    tasks being step and trace debugged.
 *
 * 5. Keeps a record of bad tasks (apart from those that have been killed) that
 *    can be examined via a /proc interface. (/proc/rtai/watchdog)
 *
 * ID: @(#)$Id: rtai_watchdog.c,v 1.3 2005/01/17 17:25:17 mdavor Exp $
 * ID: @(#)$Id: rtai_watchdog.c,v 1.3 2005/01/17 17:25:17 mdavor Exp $
 *
 *******************************************************************************/
/*static char rtai_watchdog_c_id[] __attribute__ ((unused)) = "@(#)$Id: rtai_watchdog.c,v 1.3 2005/01/17 17:25:17 mdavor Exp $";*/
/*
 * Basic rtai_watchdog module changed to meet needs by Iskra SISTEMI
 * Author: DM <davor.munda@kopica-sp.si>
 * 2004/10/07 : DM - Added max task spent uP time
 */
static char rtai_watchdog_c_id[] __attribute__ ((unused)) = "@(#)$Id: rtai_watchdog.c,v 1.3 2005/01/17 17:25:17 mdavor Exp $";

#include <linux/module.h>
#include <linux/version.h>
#include <asm/io.h>

#ifdef CONFIG_PROC_FS
#include <linux/stat.h>
#include <linux/proc_fs.h>
#include <rtai_proc_fs.h>
static struct proc_dir_entry *wd_proc;
static int    wdog_read_proc(char *page, char **start, off_t off, int count,
                             int *eof, void *data);
#endif

#include <asm/rtai.h>
#include <rtai_sched.h>
#include <rtai_watchdog.h>

/* Added by DM */
#include <rtai_fifos.h>
#include <rtai_wd.h>

// Switches on LED heartbeat and extra logging
#define WDBUG /* activated by DM */
#ifdef WDBUG
//#define LPT_PORT 0x378  /* We don't have LPT on the Target Device - by DM */
#define DBUG WDLOG
#else
#define DBUG(x...)
#endif

// Leave this defined if you don't want to use RTAI dynamic memory management
#define MY_ALLOC
#ifndef MY_ALLOC
#if defined(CONFIG_RTAI_DYN_MM) || defined(CONFIG_RTAI_DYN_MM_MODULE)
#include <rt_mem_mgr.h>
#else
#define MY_ALLOC			// Not configured so we must use our own
#endif
#endif
#ifdef MY_ALLOC
#define BAD_TASK_MAX 100		// Feel free to change this

static spinlock_t alloc_lock = SPIN_LOCK_UNLOCKED;
static BAD_RT_TASK bad_task_pool[BAD_TASK_MAX];
#endif

// The current version number
#if 0 /* Changed by DM */
static char version[] = "$Revision: 1.3 $";
#else
static char version[] = "$Revision: 1.3 $";
static RT_TASK *wd_taskP = NULL;
static RT_TASK *linux_taskP = NULL;
static RTIME start_wdtime = 0;
static int wd_idleFlag = 0, wd_priorFlag = 0, wd_badtaskFlag = 0;
#endif
static char ver[10];

// User friendly policy names
static char *policy_name[] = 
	{"Nothing", "Resync", "Debug", "Stretch", "Slip", "Suspend", "Kill"};

// Private data
static int	    num_wdogs;		// Number of watchdogs (and task lists)
static RT_TASK 	    wdog[NR_RT_CPUS];	// Watchdog tasks (1 per RT task list)
static RT_TASK 	   *tlists[NR_RT_CPUS];	// Scheduler's RT task lists
static RT_TASK	  **smp_current;	// SMP scheduler's rt_smp_current array
static BAD_RT_TASK *bad_tl[NR_RT_CPUS]; // Bad task lists (1 per watchdog)
static int          sched;		// Scheduler type (UP, SMP or MUP)

// -------------------------- CONFIGURABLE PARAMETERS --------------------------
// Module parameters
static int TickPeriod = 1000000; /*10000000; changed by DM*/	// Task period in nano seconds
MODULE_PARM(TickPeriod, "i");		// (should be shorter than all others)

static int OneShot = 0; /*1; changed by DM to periodic mode*/			// One shot timer mode or not (periodic)
MODULE_PARM(OneShot, "i");		// (should be the same as other tasks)

static int Grace = 2; /*3; changed by DM 3->2*/			// How much a task can be overdue
MODULE_PARM(Grace, "i");		// (in periods, always 1 in some modes)

static int GraceDiv = 1;		// Divisor to allow Gracevalues < 1
MODULE_PARM(GraceDiv, "i");		// overrun = period * Grace / Gracediv

static int Safety = 100;		// Safety net to suspend infinite loops
MODULE_PARM(Safety, "i");		// (overrides policy, -ve disables)

static wd_policy Policy = SUSPEND;	// How to punish misbehavers
MODULE_PARM(Policy, "i");		// (see above and header for details)

static int Stretch = 10;		// %ge to increase period by
MODULE_PARM(Stretch, "i");		// (can be over 100%, 100% is doubling)

static int Slip = 10;			// %ge of period to slip a task
MODULE_PARM(Slip, "i");			// (can be over 100%)

static int Limit = 100;			// Maximum number of offences
MODULE_PARM(Limit, "i");		// (-ve means disabled ie. no limit)

// Parameter configuring API
int rt_wdset_grace(int new)		// How much a task can be overdue
{
    int old = Grace;

    if (Policy <= STRETCH && new != 1)	return -EINVAL;
    if (new < 1) 		       	return -EINVAL;
    Grace = new;
    return old;
}

int rt_wdset_gracediv(int new)		// Divisor for Gracevalues < 1
{
    int old = GraceDiv;

    if (Policy <= STRETCH && new != 1)	return -EINVAL;
    if (new < 1) 		       	return -EINVAL;
    GraceDiv = new;
    return old;
}

int rt_wdset_safety(int new)		// Safety net to suspend infinite loops
{
    int old = Safety;

    if (new >= 0 && new < Grace)       	return -EINVAL;
    Safety = new;
    return old;
}

wd_policy rt_wdset_policy(wd_policy new) // How to punish misbehavers
{
    wd_policy old = Policy;

      /* Added by DM */
      if ( new != SUSPEND ) return -EINVAL;

    if (new < NOTHING || new > KILL)   return -EINVAL;
    if (new <= STRETCH)                Grace  = GraceDiv = 1;
    if (new == DEBUG)                  Safety = Limit = -1;
    Policy = new;
    return old;
}

int rt_wdset_slip(int new)		// %ge of period to slip a task
{
    int old = Slip;

    if (new < 0) 			return -EINVAL;
    Slip = new;
    return old;
}

int rt_wdset_stretch(int new)		// %ge to increase period by
{
    int old = Stretch;

    if (new < 0) 			return -EINVAL;
    Stretch = new;
    return old;
}

int rt_wdset_limit(int new)		// Maximum number of offences
{
    int old = Limit;

    Limit = new;
    return old;
}

// ----------------------------- MEMORY MANAGEMENT -----------------------------
static BAD_RT_TASK *new_bad_task(void)
{
#ifdef MY_ALLOC
    int		bt;

    spin_lock(&alloc_lock);
    for (bt = 0; bt < BAD_TASK_MAX; bt++) {
	if (!(bad_task_pool[bt].in_use)) {
	    bad_task_pool[bt].in_use = 1;
	    spin_unlock(&alloc_lock);
	    return &bad_task_pool[bt];
	}
    }
    spin_unlock(&alloc_lock);
    return NULL;
#else
    return rt_malloc(sizeof(BAD_RT_TASK));
#endif
}

static void free_bad_task(BAD_RT_TASK *bt)
{
#ifdef MY_ALLOC
    bt->in_use = 0;
#else
    rt_free(bt);
#endif
}

// -------------------------- LINKED LIST FUNCTIONS ----------------------------
static void append_bad_task(BAD_RT_TASK **list, BAD_RT_TASK *new)
{
    BAD_RT_TASK	*end = *list;

    if (!end) {
	*list = new;
    } else {
	while (end->next) end = end->next;
	end->next = new;
    }
}

static BAD_RT_TASK *delete_bad_task(BAD_RT_TASK **list, BAD_RT_TASK *del)
{
    BAD_RT_TASK	*rtn, *bt = *list;

    if (bt == del) {
	rtn = *list = NULL;
    } else {
	while (bt->next != del) bt = bt->next;
	rtn = bt->next = del->next;
    }
    free_bad_task(del);
    return rtn;		// Next in list
}

static BAD_RT_TASK *find_bad_task(BAD_RT_TASK *list, RT_TASK *t)
{
    BAD_RT_TASK *bt = list;

    while (bt) {
	if (bt->task == t) break;
	bt = bt->next;
    }
    return bt;
}

// ------------------------- WHICH CPU IS A TASK ON? ---------------------------
static int which_cpu(RT_TASK *t)	// Only reliable if task suspended
{
    int cpuid;

    switch (sched) {
	case UP_SCHED:  		// There is only one possibility
	    return 0;
	case MUP_SCHED: 		// Same as calling watchdog task
	    return hard_cpu_id();
	case SMP_SCHED:			// Deduce from position in list
	    for (cpuid = 0; cpuid < NR_RT_CPUS; cpuid++) {
		if (t == smp_current[cpuid]) {
		    return cpuid;
		}
	    }
	    return hard_cpu_id();	// Assume same as calling watchdog
    }
    return -1;
}

// ----------------------- SMP PROOF SUSPEND AND DELETE ------------------------
static void smpproof_task_suspend(RT_TASK *t)
{
    int cpuid;

    rt_task_suspend(t);
    if ((cpuid = which_cpu(t)) >= num_wdogs) { 		// Not really suspended
	DBUG("Resuming dummy watchdog %d\n", cpuid);
	rt_task_resume(&wdog[cpuid]);			// ...until we do this!!
    }
}

static void smpproof_task_delete(RT_TASK *t)
{
    int cpuid;

    rt_task_delete(t);
    if ((cpuid = which_cpu(t)) >= num_wdogs) {		// Not really stopped
	DBUG("Resuming dummy watchdog %d\n", cpuid);
	rt_task_resume(&wdog[cpuid]);			// ...until we do this!!
    }
}

// ----------------------------- POLICY FUNCTIONS ------------------------------
static void stretch_badtask(RT_TASK *t, BAD_RT_TASK *bt, int cpuid)
{
    // Stretch the task's period and ask scheduler to resync frame time
    t->period      += llimd(bt->orig_period, Stretch, 100);
    t->resync_frame = 1;
    DBUG( "...by %d%% to %uns\n",
	  Stretch, (int)count2nano_cpuid(t->period, cpuid));
}

static void start_slipping_badtask(RT_TASK *t, BAD_RT_TASK *bt, int cpuid)
{
    // Mark task as slipping and work out how many watchdog ticks to suspend it
    bt->slipping  = 1;
    bt->countdown = llimd( llimd(count2nano_cpuid(t->period, cpuid), Slip, 100),
	    		   1,
			   TickPeriod);
    DBUG( "Suspending task 0x%X for %d ticks (slip %d)\n",
	  t, bt->countdown, bt->count);

    // Suspend task - it will get resumed later
    smpproof_task_suspend(t);
}

static void check_slipping_badtask(BAD_RT_TASK *bt)
{
    // Resume task if it's been suspended long enough
    if (--(bt->countdown) <= 0) {
	bt->slipping = 0;
	rt_task_resume(bt->task);
	DBUG("Finished slip %d of task 0x%X, resuming\n", bt->count, bt->task);
    }
}

// ------------------------- FUNCTION TO DECIDE POLICY -------------------------
static void handle_badtask(int wd, RT_TASK *t, BAD_RT_TASK *bt, RTIME overrun)
{
    // Start 'criminal record' for first time offenders
    if (!bt) {
	bt = new_bad_task();
	if (!bt) return;
	bt->task        = t;
	bt->next        = NULL;
	bt->slipping    = 0;
	bt->count       = 0;
	bt->countdown   = 0;
	bt->valid       = 1;
	bt->forced      = 0;
	bt->orig_period = t->period;
	append_bad_task(&bad_tl[wd], bt);
    }

    // Increment offence count and note current policy
    (bt->count)++;
    bt->policy = Policy;

    // In severe cases we must suspend regardless of current policy
    if ((overrun >= (Safety * bt->orig_period)) && (Safety >= 0)) {
	WDLOG("Forcing suspension of severely overrun task 0x%X\n", t);
	bt->forced = 1;
	smpproof_task_suspend(t);
	return;
    }

    // Has it pushed its luck too far?
    if ((bt->count >= Limit) && (Limit >= 0)) {
	WDLOG("Task 0x%X reached offence limit, suspending\n", t);
	bt->forced = 1;
	smpproof_task_suspend(t);
	return;
    }

    // What to do depends on current policy
    switch (Policy) {

	case NOTHING: 	// Hope for the best
       	    break;

	case RESYNC:	// Resynchronise frame time
	case DEBUG:	// Same thing for debugging
	    WDLOG("Resynchronising task 0x%X\n", t);
	    t->resync_frame = 1;
	    break;

	case STRETCH: 	// Stretch the task's period
	    WDLOG("Stretching period of task 0x%X\n", t);
	    stretch_badtask(t, bt, wd);
	    break;

	case SLIP:    	// Suspend but arrange to resume later
	    WDLOG("Slipping task 0x%X\n", t);
	    start_slipping_badtask(t, bt, wd);
	    break;

	case SUSPEND: 	// Suspend the task
	    WDLOG("Suspending task 0x%X\n", t);
	    smpproof_task_suspend(t);
	    break;

	case KILL:    	// Delete the task
	    WDLOG("Killing task 0x%X\n", t);
	    smpproof_task_delete(t);
	    break;

	default:      	// Invalid
	    WDLOG("Invalid policy (%d)\n", Policy);
	    break;
    }
}

// -------------------------- THE MAIN WATCHDOG TASK ---------------------------
static void watchdog(int wd)
{
#if 0 /* Excluded by DM */
#ifdef WDBUG
    int 	 led    = 0;
    static int 	 output = 0x0000;
#endif
#endif
    RT_TASK 	*task, *self = rt_whoami();
    BAD_RT_TASK *bt;
    RTIME 	 now, overrun;
    int		 another, dog;

      /* Added by DM */
      RTIME start_criticaltime = 0;
      RTIME val_idletime, diff_time;
      RT_TASK *diag_task;
      int firstLoop = 0;

    while (1) {
#if 0 /* Excluded by DM */
#ifdef WDBUG
	// LED heartbeat visible on parallel port
	led = !led;
	if (led) output |=  (1 << wd);
	else     output &= ~(1 << wd);
	outb(output, LPT_PORT);
#endif
#endif
	// Fix any overrun of our own (step and trace debug in oneshot mode)
	now = rt_get_time_cpuid(wd);
	if (now - self->resume_time >= self->period) {
	    self->resync_frame = 1;
	    rt_task_wait_period();
	    DBUG("Resynchronised watchdog %d\n", wd);
	    continue;
	}

	// Mark all this watchdog's bad tasks as invalid ready for check
	for (bt = bad_tl[wd]; bt; bt->valid = 0, bt = bt->next);

	// Loop through all the RT tasks in this watchdog's list
	task = tlists[wd];

	  /* Added by DM */
	  if ( !firstLoop ) { /* Set all uP times to 0 */
	      for ( diag_task = task; diag_task; diag_task = diag_task->next ) {
		      diag_task->add_tinfo.sum_uptime = 0;
	          diag_task->add_tinfo.max_uptime = 0;
	      }
	      start_wdtime = now;
	      wd_taskP = self;
	      linux_taskP = task;
	      firstLoop = 1;
          }
	  else { /* Check Idle (Linux) task time */
              val_idletime = (task->add_tinfo.sum_uptime << 2) + task->add_tinfo.sum_uptime;
	      diff_time = now - start_wdtime;
	      if ( val_idletime < diff_time  &&  !start_criticaltime ) {
                  start_criticaltime = now;
	      }
	      else {
                  if ( val_idletime < diff_time  &&  start_criticaltime
		       &&
		       count2nano(now - start_criticaltime) > 1000000000 /* 1 sec */ ) {
	              WDLOG("Idle (Linux) task has critical low process time\n");
		      for ( diag_task = tlists[wd]->next; diag_task; diag_task = diag_task->next ) {
                      /* Suspend RT tasks (leave Linux and WD) */
		          if ( diag_task == self  ||  (diag_task->state & SUSPENDED) ) continue;
                          smpproof_task_suspend(diag_task);
		      }
                      start_criticaltime = 0;
		      wd_idleFlag = 1;
		  }
		  else {
		      if ( val_idletime >= diff_time  &&  start_criticaltime ) {
		          start_criticaltime = 0;
		      }
		  }
	      }
	  }

	while ((task = task->next)) {

	    // Ignore ourself but note another watchdog
	    if (task == self) continue;
	    for (another = dog = 0; dog < num_wdogs; dog++) {
		if (task == &wdog[dog]) {
		    another = 1 + dog;
		    break;
		}
	    }

	    // Search for any criminal record and check slipping tasks
	    if ((bt = find_bad_task(bad_tl[wd], task))) {
		bt->valid = 1;
		if (bt->slipping) {
		    check_slipping_badtask(bt);
		    continue;
		}
	    }

	      /* Added by DM */
	      if ( !(task->state & SUSPENDED)
	           &&
		   (task->priority <= self->priority
		    /*|| allow other periods
		    (task->period  &&  task->period <= self->period)*/) ) {
		  /*WDLOG("Found task 0x%X (list %d) with Priority or Period over WD\n", task, wd);*/
		  WDLOG("Found task 0x%X (list %d) with Priority over WD\n", task, wd);
		  for ( diag_task = tlists[wd]->next; diag_task; diag_task = diag_task->next ) {
	          /* Suspend RT tasks (leave Linux and WD) */
		      if ( diag_task == self  ||  (diag_task->state & SUSPENDED) ) continue;
                      smpproof_task_suspend(diag_task);
		  }
                  wd_priorFlag = 1;
	      }

	    // Ignore non-periodic, resyncing, suspended or blocked tasks
	    if (!task->period || task->resync_frame || task->state &
		    (SUSPENDED|DELAYED|SEMAPHORE|SEND|RECEIVE|RPC|RETURN)) {
		continue;
	    }

	    // Check for overrun and decide what to do (ignore other watchdogs)
		 overrun = now - task->resume_time;

		 overrun = overrun >= 0 ? overrun : -overrun; /* added by DM, we need positive value */

		  if (overrun >= llimd(task->period, Grace, GraceDiv)
		      && task->add_tinfo.sum_uptime != 0/* added by DM to ensure first cycle */) {
		   if (another--) {
		    WDLOG("WARNING: Watchdog %d is overrunning: duration=%Ldns\n", another, count2nano(overrun));
		   } else {
			WDLOG("Found overrunning task 0x%X (list %d): duration=%Ldns\n", task, wd, count2nano(overrun));
		    handle_badtask(wd, task, bt, overrun);

		      /* Added by DM */
		      for ( diag_task = tlists[wd]->next; diag_task; diag_task = diag_task->next ) {
	              /* Suspend other RT tasks (leave Linux and WD) */
			    if ( diag_task == self  ||  diag_task == task  ||  (diag_task->state & SUSPENDED) ) continue;
			    overrun = now - diag_task->resume_time; /* Info about others overrunning tasks */
		        overrun = overrun >= 0 ? overrun : -overrun;
		        if (overrun >= llimd(diag_task->period, Grace, GraceDiv)
				    && diag_task->add_tinfo.sum_uptime != 0) {
			        WDLOG("Found overrunning task 0x%X (list %d): duration=%Ldns\n", diag_task, wd, count2nano(overrun));
                }
				smpproof_task_suspend(diag_task);
		      }
		      wd_badtaskFlag = 1;
           }
		  }
	}

	// Clean up any bad tasks still marked invalid (their RT task has gone)
	for (bt = bad_tl[wd]; bt;) {
	    if (!(bt->valid)) {
		bt = delete_bad_task(&bad_tl[wd], bt);
	    } else {
		bt = bt->next;
	    }
	}

	// Wait for next watchdog 'tick'
	rt_task_wait_period();
    }
}

// -------------------------- DUMMY WATCHDOG TASK ------------------------------
static void dummy(int wd)
{
    // Go straight back to sleep - see SMP proof suspend and delete
    while (1) {
	rt_task_suspend(&wdog[wd]);
    }
}

// ----------------------------- CMD FIFO Handler by DM -----------------------
static SEM  fifo_sem;
static int wd_cmd_fifo_handler( unsigned int  fifoId )
{
 RT_TASK      *task;
 WD_CMD        cmd;
 WD_TASK_DATA  data;
 int           res, tcount=0, tflag=0;


    if ( (res = rtf_get( fifoId, &cmd, sizeof(cmd) )) == sizeof(cmd) ) {
        switch ( cmd.cmd ) {
	    case WD_RESET_DATA_CMD :
	    /* Reset Data FIFO */
		rt_sem_wait( &fifo_sem );
                rtf_reset( WD_DATA_FIFO );
                rt_sem_signal( &fifo_sem );
		break;

	    case WD_GET_DATA_CMD :
	    /* Put Data */
		if ( (task = linux_taskP) == NULL ) break;
		rt_sem_wait( &fifo_sem );
                for ( ; task; task = task->next ) {
		    if ( ++tcount > WD_RTTASK_MAX_NO  &&  !tflag ) { /* To many RT Tasks */
                        memcpy(data.name, WD_OVERFLOW_TASKNAME, sizeof(data.name));
			data.handle = 0;
                        data.uptime.sec = data.uptime.ns = 0;
						data.max_uptime = 0;
			tflag = 1;
		    }
		    else
		        if ( tcount <= WD_RTTASK_MAX_NO ) {
			    if ( task == linux_taskP ) memcpy(data.name, WD_LINUX_TASKNAME, sizeof(data.name));
			    else if ( task == wd_taskP ) memcpy(data.name, WD_SELF_TASKNAME, sizeof(data.name));
			         else memcpy(data.name, WD_OTHER_TASKNAME, sizeof(data.name));
		            data.handle = (unsigned long)task;
                            data.uptime.sec = ulldiv( count2nano(task->add_tinfo.sum_uptime),
			                                         NSECS_PER_SEC,
			                                         &data.uptime.ns);
							data.max_uptime = (unsigned long)count2nano(task->add_tinfo.max_uptime);
                        }
		    if ( tcount <= WD_RTTASK_MAX_NO + 1 /* Put Data into the FIFO */
			 &&
			 rtf_put( WD_DATA_FIFO, &data, sizeof(data) ) != sizeof(data) ) {
	                WDLOG("Put Data FIFO error\n");
		    }
		} /* for */
		/* Append last record */
                memcpy(data.name, WD_LAST_RECORDNAME, sizeof(data.name));
		data.handle = 0;
                data.uptime.sec = data.uptime.ns = 0;
				data.max_uptime = 0;
		if ( rtf_put( WD_DATA_FIFO, &data, sizeof(data) ) != sizeof(data) )
	            WDLOG("Put Data FIFO error\n");
		rt_sem_signal( &fifo_sem );
		break;

	    case WD_GET_RTSTATUS_CMD :
            /* Put Status */
		if ( wd_idleFlag  || wd_priorFlag  ||  wd_badtaskFlag ) {
                    memcpy(data.name, WD_RT_STATUSNAME, sizeof(data.name));
		    data.handle = (unsigned long)WD_SUSPEND_STATUS;
                    data.uptime.sec = data.uptime.ns = 0;
					data.max_uptime = 0;
		}
		else {
                    memcpy(data.name, WD_RT_STATUSNAME, sizeof(data.name));
		    data.handle = (unsigned long)WD_RUN_STATUS;
                    data.uptime.sec = data.uptime.ns = 0;
					data.max_uptime = 0;
		}
		rt_sem_wait( &fifo_sem );
		if ( rtf_put( WD_DATA_FIFO, &data, sizeof(data) ) != sizeof(data) )
	            WDLOG("Put Data FIFO error\n");
		rt_sem_signal( &fifo_sem );
		break;

	    case WD_RESET_UPTIMES_CMD :
            /* Reset Processor Times for all tasks */
		if ( (task = linux_taskP) == NULL ) break;
                for ( ; task; task = task->next ) {
	            task->add_tinfo.sum_uptime = 0;
                task->add_tinfo.max_uptime = 0;
	        }
		start_wdtime = rt_get_time();
		break;

	    default :
	        WDLOG("Invalid CMD FIFO command\n");
		break;
	}
    }
    else
        if ( res != 0 )
    	    WDLOG("Get Cmd FIFO error\n");


   return res;
}

// ----------------------------- MODULE CONTROL --------------------------------
int init_module(void)
{
    RTIME 	 period;
    int		 dog;
    RT_TASK	*lnx0;
    struct apic_timer_setup_data apic_data[NR_RT_CPUS];
    char	*c;

      /* Added by DM */
      if ( Policy != SUSPEND
           ||
	   rt_sched_type() != UP_SCHED ) {
	  WDLOG("Must set SUSPEND policy with UP Scheduler\n");
	  return 1;
      }
      rt_sem_init( &fifo_sem, 1 );
      rtf_reset( WD_CMD_FIFO );
      rtf_reset( WD_DATA_FIFO );
      if ( rtf_create( WD_CMD_FIFO, WD_CMD_FIFO_SIZE ) ) {
	  WDLOG("Command FIFO creation error\n");
          rt_sem_delete( &fifo_sem );
	  return 1;
      }
      if ( rtf_create( WD_DATA_FIFO, WD_DATA_FIFO_SIZE ) ) {
	  WDLOG("DATA FIFO creation error\n");
          rt_sem_delete( &fifo_sem );
	  rtf_destroy( WD_CMD_FIFO );
	  return 1;
      }
      if ( rtf_create_handler( WD_CMD_FIFO, wd_cmd_fifo_handler ) ) {
	  WDLOG("CMD FIFO handler error\n");
	  rtf_destroy( WD_CMD_FIFO );
	  rtf_destroy( WD_DATA_FIFO );
          rt_sem_delete( &fifo_sem );
	  return 1;
      }

    // Some parameters have to be forced
    if (Policy <= STRETCH) Grace  = GraceDiv = 1;
    if (Policy == DEBUG)   Safety = Limit = -1;

    // Deduce number of watchdogs needed from scheduler type
    switch (sched = rt_sched_type()) {
	case UP_SCHED  : 			 // Fall through
	case SMP_SCHED : num_wdogs = 1;          break;
	case MUP_SCHED : num_wdogs = NR_RT_CPUS; break;
    }

    // Fill array of pointers to scheduler's task lists
    lnx0 = rt_get_base_linux_task(tlists);

    // Register watchdogs with scheduler (SMP returns pointer to rt_smp_current)
    for (dog = 0; dog < NR_RT_CPUS; dog++) {
	if ((smp_current = rt_register_watchdog(&wdog[dog], dog)) < 0) {
	    WDLOG("Failed to register watchdog %d with RTAI scheduler\n", dog);
	    for (dog--; dog >= 0; dog--) rt_deregister_watchdog(&wdog[dog], dog);
	    return -EBUSY;
	}
    }

    // Set up chosen timer mode - MUP lets you have different modes per CPU,
    // but you'll have to edit the code below a bit if that's what you want.
    if (sched == MUP_SCHED) {
	for (dog = 0; dog < num_wdogs; dog++) {
	    apic_data[dog].mode  = !OneShot;	 // <--- This bit...
	    apic_data[dog].count = TickPeriod;
	    if (OneShot) {
		rt_preempt_always_cpuid(1, dog); // <--- ...and this!
	    }
	}
	start_rt_apic_timers(apic_data, 0);
    } else {
#if 0 /* Excluded by DM - we already have the timer in the system */
	if (OneShot) {
	    rt_set_oneshot_mode();
	    rt_preempt_always(1);
	} else {
	    rt_set_periodic_mode();
	}
	start_rt_timer((int)nano2count(TickPeriod));
#endif
    }

    // Set up and start watchdog tasks (on separate CPUs if MP). We run as
    // many real watchdogs as there are task lists. However we must protect
    // the remaining CPUs with dummy watchdogs to prevent them being hogged
    // by overrunning tasks (only relevant on SMP not MUP).
    for (dog = 0; dog < NR_RT_CPUS; dog++) {
	rt_task_init_cpuid( 	&wdog[dog],
			    	(dog < num_wdogs) ? watchdog : dummy,
			    	dog, 2000, RT_HIGHEST_PRIORITY, 0, 0, dog);
    }
    for (dog = 0; dog < num_wdogs; dog++) {
	period = nano2count_cpuid(TickPeriod, dog);
	rt_task_make_periodic( 	&wdog[dog],
			       	rt_get_time_cpuid(dog) + period,
			       	period);
    }

    // Tidy up version number
    if ((c = strchr(version, ' '))) {
	*(strchr(c, '$')) = '\0';
	strcpy(ver, c + 1);
    } else {
	strcpy(ver, "? ");
    }

    // Log initial parameters
    WDLOG( "\n==== RTAI Watchdog v%s(%s, %s) Loaded ====\n",
	   ver, __TIME__, __DATE__);
    WDLOG( "%d Watchdog task%s running @ %dHz in %s mode\n",
	   num_wdogs, num_wdogs > 1 ? "s" : "",
	   imuldiv(NSECS_PER_SEC, 1, TickPeriod),
	   OneShot ? "oneshot" : "periodic");
#ifdef MY_ALLOC
    WDLOG( "Using static memory management (%d entries)\n", BAD_TASK_MAX);
#else
    WDLOG( "Using dynamic memory management\n");
#endif
    WDLOG( "Policy         : '%s'\n", policy_name[Policy]);
    WDLOG( "Grace periods  : %d%s\n", Grace,
	   (Policy <= STRETCH) ? " (forced)" : "");
    WDLOG( "Grace divisor  : %d%s\n", GraceDiv,
	   (Policy <= STRETCH) ? " (forced)" : "");
    WDLOG( "Safety limit   : ");
    if (Safety < 0) {
	rt_printk("(disabled)\n");
    } else {
	rt_printk("%d period%s\n", Safety, Safety > 1 ? "s" : " ");
    }
    WDLOG( "Slip factor    : %d%%\n", Slip);
    WDLOG( "Stretch factor : %d%%\n", Stretch);
    WDLOG( "Offense limit  : ");
    if (Limit < 0) {
	rt_printk("(disabled)\n");
    } else {
	rt_printk("%d\n", Limit);
    }

#ifdef CONFIG_PROC_FS
    // Register /proc interface
    wd_proc = create_proc_entry("watchdog", 0, rtai_proc_root);
    wd_proc->read_proc = wdog_read_proc;
#endif
    return 0;
}

void cleanup_module(void)
{
    BAD_RT_TASK	*bt;
    int		 dog;

#ifdef CONFIG_PROC_FS
    // Remove /proc interface
    remove_proc_entry("watchdog", rtai_proc_root);
#endif
    // Deregister all watchdogs and shutdown the timer
    for (dog = 0; dog < NR_RT_CPUS; dog++) {
	rt_deregister_watchdog(&wdog[dog], dog);
    }
#if 0 /* Excluded by DM - we already have the timer in the system */
    stop_rt_timer();
    rt_busy_sleep(TickPeriod);
#endif

    // Cleanup and remove all watchdogs and bad task lists
    for (dog = 0; dog < NR_RT_CPUS; dog++) {
	rt_task_delete(&wdog[dog]);
	if (dog < num_wdogs) {
	    for (bt = bad_tl[dog]; bt;) {
		bt = delete_bad_task(&bad_tl[dog], bt);
	    }
	}
    }

      /* Added by DM */
      rtf_destroy( WD_CMD_FIFO );
      rtf_destroy( WD_DATA_FIFO );
      rt_sem_delete( &fifo_sem );

    // It's all over :(
    WDLOG("==== RTAI Watchdog v%sUnloaded ====\n", ver);
}

// ----------------------------- PROC INTERFACE --------------------------------
#ifdef CONFIG_PROC_FS
static int wdog_read_proc(char *page, char **start, off_t off, int count,
                          int *eof, void *data)
{
    PROC_PRINT_VARS;
    RT_TASK	*task;
    BAD_RT_TASK	*bt;
    long 	 onsec, osec;
    long 	 ansec, asec;
    int		 cpuid, tl, id = 1;
    char	 action[10];

    // Heading and parameters
#if 0 /* Changed by DM */
    PROC_PRINT("\nRTAI Watchdog Status\n");
    PROC_PRINT(  "--------------------\n");
#else
    PROC_PRINT("\nRTAI (Iskra SISTEMI) Watchdog Status\n");
    PROC_PRINT(  "------------------------------------\n");
#endif
    PROC_PRINT("%d Watchdog task%s running @ %dHz in %s mode\n", 
	       num_wdogs, num_wdogs > 1 ? "s" : "",
	       imuldiv(NSECS_PER_SEC, 1, TickPeriod),
	       OneShot ? "oneshot" : "periodic");
#ifdef MY_ALLOC
    PROC_PRINT("Using static memory management (%d entries)\n", BAD_TASK_MAX);
#else
    PROC_PRINT("Using dynamic memory management\n");
#endif
    PROC_PRINT("Policy         : '%s'\n", policy_name[Policy]);
    PROC_PRINT("Grace periods  : %d%s\n", Grace, 
	       (Policy <= STRETCH) ? " (forced)" : "");
    PROC_PRINT("Grace divisor  : %d%s\n", GraceDiv, 
	       (Policy <= STRETCH) ? " (forced)" : "");
    PROC_PRINT("Safety limit   : ");
    if (Safety < 0) {
	PROC_PRINT("(disabled)\n");
    } else {
	PROC_PRINT("%d period%s\n", Safety, Safety > 1 ? "s" : "");
    }
    PROC_PRINT("Slip factor    : %d%%\n", Slip);
    PROC_PRINT("Stretch factor : %d%%\n", Stretch);
    PROC_PRINT("Offense limit  : ");
    if (Limit < 0) {
	PROC_PRINT("(disabled)\n");
    } else {
	PROC_PRINT("%d\n", Limit);
    }

    // List criminal records
    PROC_PRINT("\nBad tasks...\n\n");
    PROC_PRINT("RT Task    ID "
	       "CPU%s "
	       "Priority State Count "
	       "Original period Adjusted period "
	       "Action\n",
	       (sched == SMP_SCHED) ? "s" : "");
    PROC_PRINT("---------- -- "
	       "---%s "
	       "-------- ----- ----- "
	       "--------------- --------------- "
	       "---------\n",
	       (sched == SMP_SCHED) ? "-" : "");
    for (tl = 0; tl < num_wdogs; tl++) {
	task = tlists[tl];
	while ((task = task->next)) {
	    if ((bt = find_bad_task(bad_tl[tl], task))) {
		if (bt->forced) {
		    sprintf(action, "%s *", policy_name[SUSPEND]);
		} else {
		    strcpy(action, policy_name[bt->policy]);
		}
		cpuid = (sched == MUP_SCHED) ? task->runnable_on_cpus : 0;
		osec  = ulldiv( count2nano_cpuid(bt->orig_period, cpuid),
			        NSECS_PER_SEC, 
			        &onsec);
		asec  = ulldiv( count2nano_cpuid(task->period, cpuid),
			        NSECS_PER_SEC, 
			        &ansec);
		PROC_PRINT( "0x%08x %-2d "
			    "%s%-2d%s "
			    "%-8d 0x%-3x %-5d "
			    "%02ds %09dns %02ds %09dns "
			    "%s\n",
			    (int)task, id, 
			    (sched == SMP_SCHED) ?  "0x" : "",
			    (sched == UP_SCHED)  ?
			        0 : (int)task->runnable_on_cpus,
			    (sched == SMP_SCHED) ?  ""   : " ",
			    task->priority, task->state, bt->count,
			    (int)osec, (int)onsec, (int)asec, (int)ansec, 
			    action);
	    }
	    id++;
	}
    }

      /*  Added by DM */
      if ( wd_idleFlag ) {
	  PROC_PRINT(  "------------------------------------\n");
	  PROC_PRINT(  "Critical idle (Linux) task time. RT tasks were suspended\n");
      }
      if ( wd_priorFlag ) {
          PROC_PRINT(  "------------------------------------\n");
	  PROC_PRINT(  "Too high task priority or period. RT tasks were suspended\n");
      }
      if ( wd_badtaskFlag ) {
          PROC_PRINT(  "------------------------------------\n");
	  PROC_PRINT(  "Overrunning task. RT tasks were suspended\n");
      }

    PROC_PRINT_DONE;
}
#endif
