/***************************************************************************
  PROJECT    : RTAI-Watchdog / wd.cpp
  DESCRIPTION: WD user API
  AUTHOR     : DM <davor.munda@kopica-sp.si>

  DM         2003/06/07    1.1.0    Basic version
 ***************************************************************************/

#include <iostream.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include "wd.h"


wd_class::wd_class()
// constructor
{
    if ((cmdFd = open(WD_CMD_FIFO_NAME, O_WRONLY)) < 0) {
        fprintf(stderr, "Error opening %s errno=%d\n", WD_CMD_FIFO_NAME, errno);
        exit(1);
    }

    if ((dataFd = open(WD_DATA_FIFO_NAME, O_RDONLY)) < 0) {
	close( cmdFd );
        fprintf(stderr, "Error opening %s errno=%d\n", WD_DATA_FIFO_NAME, errno);
        exit(1);
    }

    // O.K.
    return;
}


wd_class::~wd_class()
// destructor
{
    // Close RTFs
    close( dataFd );
    close( cmdFd );

    return;
}


WD_READ_DATA_RESULT_ENUM wd_class::read_data( int           &no_tasks,
                                              WD_TASK_DATA  &data,
                                              const int     space_records )
// read RT tasks data
{
    fd_set          rfds;
    struct timeval  tv;
    int             retval;
    int             n, i;
    WD_CMD          cmd;
    char           *dPtr = (char *)&data;
    WD_TASK_DATA    dBuff;


    // reset FIFO
    cmd.cmd = WD_RESET_DATA_CMD;
    if ( write_buffer(cmdFd, &cmd, sizeof(cmd)) < 0 ) {
        return WD_READ_DATA_ERR;
    }

    // data request
    cmd.cmd = WD_GET_DATA_CMD;
    if ( write_buffer(cmdFd, &cmd, sizeof(cmd)) < 0 ) {
        return WD_READ_DATA_ERR;
    }

    FD_ZERO(&rfds);
    FD_SET(dataFd, &rfds);

    // Set ms timeout
    tv.tv_sec = 0;
    tv.tv_usec = 1000000; // 1 sec

    // Block mode wait
    retval = select( FD_SETSIZE, &rfds, NULL, NULL, &tv );

    if ( retval > 0
         &&
         FD_ISSET( dataFd, &rfds ) ) {
	// Read task records
        for ( i = 0; i < space_records + 1; ++i ) {
	    if ( (n = read_buffer( dataFd, &dBuff, sizeof(dBuff) )) != sizeof(dBuff) ) {
		return WD_READ_DATA_ERR; // error
            }
	    else {
		if ( !memcmp(dBuff.name, WD_LAST_RECORDNAME, sizeof(dBuff.name)) ) break; // TheEnd
   	        memcpy(dPtr, &dBuff, sizeof(dBuff));
		dPtr += sizeof(dBuff);
	    }
        } /* for */

	if ( i == space_records + 1 )
	    return WD_READ_DATA_MORE; // we have more data - increase buffer


	// O.K.
        no_tasks = i;
        return WD_READ_DATA_OK;
    } /* if */


    // Timeout
    return WD_READ_DATA_ERR;
}


WD_RTSTATUS_ENUM wd_class::read_rtstatus( void )
// read RT tasks status
{
    fd_set          rfds;
    struct timeval  tv;
    int             retval;
    int             n;
    WD_CMD          cmd;
    WD_TASK_DATA    dBuff;


    // reset FIFO
    cmd.cmd = WD_RESET_DATA_CMD;
    if ( write_buffer(cmdFd, &cmd, sizeof(cmd)) < 0 ) {
        return WD_READSTATUS_ERR;
    }

    // status request
    cmd.cmd = WD_GET_RTSTATUS_CMD;
    if ( write_buffer(cmdFd, &cmd, sizeof(cmd)) < 0 ) {
        return WD_READSTATUS_ERR;
    }

    FD_ZERO(&rfds);
    FD_SET(dataFd, &rfds);

    // Set ms timeout
    tv.tv_sec = 0;
    tv.tv_usec = 1000000; // 1 sec

    // Block mode wait
    retval = select( FD_SETSIZE, &rfds, NULL, NULL, &tv );

    if ( retval > 0
         &&
         FD_ISSET( dataFd, &rfds ) ) {
	// Read status record
        if ( (n = read_buffer( dataFd, &dBuff, sizeof(dBuff) )) != sizeof(dBuff) ) {
	    return WD_READSTATUS_ERR; // error
        }
	else {
	    if ( memcmp(dBuff.name, WD_RT_STATUSNAME, sizeof(dBuff.name)) ) return WD_READSTATUS_ERR;
	    return (WD_RTSTATUS_ENUM)dBuff.handle;
	}
    } /* if */


    // Timeout
    return WD_READSTATUS_ERR;
}


WD_RESETTIME_RESULT_ENUM wd_class::reset_uptimes( void )
// reset uP times to 0
{
    WD_CMD          cmd;


    // reset uP times request
    cmd.cmd = WD_RESET_UPTIMES_CMD;
    if ( write_buffer(cmdFd, &cmd, sizeof(cmd)) < 0 ) {
        return WD_RESETTIME_ERR;
    }


    // O.K.
    return WD_RESETTIME_OK;
}



int  wd_class::write_buffer(int fd, const void *buf, const int count)
{
 const void *pts = buf;
 int         status = 0, n;

    while ( status != count )
    {
        n = write(fd, (void *)((unsigned long)pts + status), count - status);
        if ( n < 0 ) return n;
        status += n;
    }

    return status;
}


int  wd_class::read_buffer(int fd, void *buf, const int count)
{
 void *pts = buf;
 int   status = 0, n;

    while ( status != count )
    {
        n = read(fd, (void *)((unsigned long)pts + status), count - status);
        if ( n < 1 ) return n;
        status += n;
    }

    return status;
}

