/*****
 *
 * Iskra SISTEMI (C) 2003  DM (davor.munda@kopica-sp.si)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Module: rtx_rtai.c - RTX Interface.
 *
 * Version: 1.1.0 - 2003/04/01 : Basic version - DM
 * Version: 1.2.0 - 2003/04/14 : The same API for LXRT and kernel Hard RT, added alloc routines - DM
 * Version: 2.1.0 - 2003/05/06 : Build library in UserSpace and module in Kernel - DM
 * Version: 2.2.0 - 2003/05/20 : Remove timers interface (we use only one global timer) - DM
 * Version: 2.3.0 - 2003/06/09 : Include rtai_wd.h Watchdog header and use Watchdog limits - DM
 * Version: 2.4.0 - 2004/12/08 : Use periodic timer instead of oneshot one in the kernel - DM
 * Version: 2.4.1 - 2005/10/25 : Start periodic tasks only one per timer period - DM
 * Version: 2.5.0 - 2005/11/22 : Win32 RTOS port - DM
 *
 *****/


#ifdef WIN32
 #include <os_api.h>
#endif

#ifdef OS_WIN32
 #include <rtx.h>
 #include <direct.h>
 #ifndef __KERNEL__
  #define __KERNEL__
 #endif
#else

#ifdef __KERNEL__   /* kernel export table */
#ifndef EXPORT_SYMTAB
 #define EXPORT_SYMTAB
#endif
#endif
#endif


#include <rtx_types.h>


#define _DBG_RTX_    /* Debug output */

#ifndef _DBG_RTX_
/* Don't print messages */
#define  printx_(msg)       ;
#define  printx__(msg,arg)  ;
#else
/* Print messages */
#define  printx_   printx
#define  printx__  printx
#endif



#ifndef  __KERNEL__
#include <rtai_lxrt_user.h>
#include <rtai_lxrt.h>
#include <rtai_fifos_lxrt.h>
#else
#define  SYS_TICK_PERIOD  1000000  /* 1 ms base system period */
static int PeriodicTickBase = SYS_TICK_PERIOD;
#ifndef OS_WIN32
MODULE_PARM(PeriodicTickBase, "i");
 #ifdef __arm__
  extern long rtai_hz;
  extern long rtai_tick;
  extern long rtai_latch;
  extern unsigned long rtai_tick_cnt;
 #endif
#else
 #define MAX_FIFOS 64
 static char sRtFifoNames[MAX_FIFOS][100];
 static long sRtFifoSeek[MAX_FIFOS];
 static FILE *sRtFifoHnd[MAX_FIFOS];
 static TOs_Hnd sRtFifoHandle[MAX_FIFOS];
#endif
#endif



static  int no_rtx_tasks = 0;  /* tasks counter */

#ifdef OS_WIN32
HANDLE hMutex; /* enable/disable FIFO mutex */
//CRITICAL_SECTION criticalSection;
#endif



/* #########################################################################
    RTX  Functions
   ######################################################################### */

/**
    Functions
**/

#ifndef __KERNEL__
RTX_RESULT  runThread( rtx_task_handlePtr        *hndPtrP,       /* handle pointer to pointer */
                       const rtx_priority         priority,      /* task priority */
                       const rtx_task_stacksize   stack,         /* stack size - bytes */
                       const rtx_task_mode        mode,          /* task mode */
                       const rtx_periodic_tick    nanoTick,      /* tick for periodic mode - nano sec */
                       rtx_periodic_tickPtr       tickPtr        /* tick space pointer */ );
RTX_RESULT  killThread( rtx_thread_handle  hnd /* handle */ );
RTX_RESULT  getMBX( rtx_mbx_handlePtr     *mbxChPtrP, /* channel pointer to pointer */
                    const rtx_mbx_msgLen   msgLen,    /* message length - bytes */
                    const rtx_mbx_chMsgs   chMsgs     /* max no messages in the channel */ );
RTX_RESULT  rtfCreateHnd( const rtx_fifo_id              fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                          rtx_fifo_handler_lxrt_thread   fifoHandler,/* LXRT thread - must be eternity loop and must poll fifo */
						  rtx_thread_handlePtr           thread_hndPtr/* thread handle pointer */ );
#else
RTX_RESULT  getMBX( rtx_mbx_handlePtr      mbxChPtr, /* channel pointer */
                    const rtx_mbx_msgLen   msgLen,   /* message length - bytes */
                    const rtx_mbx_chMsgs   chMsgs    /* max no messages in the channel */ );
RTX_RESULT  rtfCreateHnd( const rtx_fifo_id  fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
#ifndef OS_WIN32
                          rtx_fifo_handler   fifoHandler /* when Linux process (user space) accesses fifo - R/W, handler routine is called */ 
#else
						  rtx_fifo_handler_win32_thread   fifoHandler, /* Win32 thread - must be eternity loop and must poll fifo */
						  rtx_thread_handlePtr            thread_hndPtr/* fake thread handle pointer */
#endif
						);
#endif


RTX_RESULT  XRunTask( rtx_task_handlePtr        hndPtr,       /* handle pointer */
                      const rtx_priority        priority,     /* task priority */
                      const rtx_task_stacksize  stack,        /* stack size - bytes */
                      const rtx_task_mode       mode,         /* task mode */
                      const rtx_periodic_tick   nanoTick,     /* tick for periodic mode - nano sec */
                      rtai_task                 task_entry,   /* entry point */
                      rtx_thread_handlePtr      thread_hndPtr,/* thread handle pointer */
                      const rtx_task_arg        arg           /* argument */ );
RTX_RESULT  XKillTask( rtx_task_handlePtr  hndPtr,   /* handle pointer */
                       rtx_thread_handle   hndThread /* thread handle */ );
void  XExit( void );
rtx_task_handlePtr  XGetId( void );
void  XLock( void );
void  XUnLock( void );
RTX_RESULT  XCPri( rtx_task_handlePtr  hndPtr,  /* handle pointer */
                   const rtx_priority  priority /* priority */ );
void  XDelay( rtx_periodic_tickno         ticks,  /* no ticks */
              const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );
void  XDelayShort( rtx_nano_time  ntime /* no nano secs */ );
RTX_RESULT  XStop( rtx_task_handlePtr  hndPtr /* handle pointer */ );
RTX_RESULT  XResume( rtx_task_handlePtr  hndPtr /* handle pointer */ );
rtx_periodic_tick  XGetTickInterv( const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );
RTX_RESULT  XSetTickInterv( rtx_periodic_tickPtr     tickPtr,  /* tick pointer */
                            const rtx_periodic_tick  nanoTick  /* tick for periodic mode - nano sec */ );
rtx_periodic_tickno  XGetNoTicksTrunc( const rtx_nano_time         ntime,  /* no nano secs */
                                       const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );
rtx_periodic_tickno  XGetNoTicksRoundUp( const rtx_nano_time         ntime,  /* no nano secs */
                                         const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );
rtx_periodic_tickno  XGetNoTicks( const rtx_nano_time         ntime,  /* no nano secs */
                                  const rtx_periodic_tickPtr  tickPtr /* tick pointer */ );

RTX_RESULT  XReleaseMBX( rtx_mbx_handlePtr  mbxChPtr /* channel pointer */ );
RTX_RESULT  XSendMBX( rtx_mbx_handlePtr         mbxChPtr,      /* channel pointer */
                      const rtx_mbx_msgPtr      mbxMsgPtr,     /* message pointer */
                      const rtx_mbx_msgLen      msgLen,        /* message length - bytes */
                      const rtx_nano_time       busyWait,      /* timeout if channel is bussy - nano secs - 0 don't wait */
                      const rtx_nano_time       readTaskWait,  /* timeout to be read and deleted by task - nano secs - 0 don't wait */
                      const rtx_task_handlePtr  readTaskHndPtr /* read task handle pointer */ );
RTX_RESULT  XWaitMBX( rtx_mbx_handlePtr     mbxChPtr,  /* channel pointer */
                      rtx_mbx_msgPtr        mbxMsgPtr, /* message pointer */
                      const rtx_mbx_msgLen  msgLen,    /* message length - bytes */
                      const rtx_nano_time   arriveWait /* timeout if channel is empty - nano secs */ );
RTX_RESULT  XReadMBX( rtx_mbx_handlePtr         mbxChPtr,       /* channel pointer */
                      rtx_mbx_msgPtr            mbxMsgPtr,      /* message pointer */
                      const rtx_mbx_msgLen      msgLen,         /* message length - bytes */
                      const rtx_nano_time       arriveWait,     /* timeout if channel is empty - nano secs - 0 don't wait */
                      const rtx_task_handlePtr  replyTaskHndPtr /* reply task handle pointer - NULL don't reply */ );

RTX_RESULT  XGetFreeMuteX( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );
RTX_RESULT  XReleaseMuteX( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );
RTX_RESULT  XLockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );
RTX_RESULT  XUnlockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );
RTX_RESULT  XTryLockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ );

RTX_RESULT  XRtfCreate( const rtx_fifo_id     fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                        const rtx_fifo_size   fifoSize,   /* rtf fifo size - bytes */
                        rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                        rtx_fifo_sizePtr      fifoSizePtr /* rtf fifo size space pointer */ );
void  XRtfDestroy( const rtx_fifo_id     fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                   rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                   rtx_fifo_sizePtr      fifoSizePtr,/* rtf fifo size space pointer */
				   rtx_thread_handlePtr  thread_hndPtr/* thread handle pointer */ );
RTX_RESULT  XRtfReset( const rtx_fifo_id     fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                       rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                       rtx_fifo_sizePtr      fifoSizePtr /* rtf fifo size space pointer */ );
rtx_fifo_okBytes  XRtfPut( const rtx_fifo_id        fifoId,    /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                           const rtx_fifo_dataPtr   dataPtr,   /* data pointer */
                           const rtx_fifo_dataSize  dataSize,  /* data size - bytes */
                           rtx_fifo_devhandlePtr    fifoDevPtr /* rtf fifo device pointer */ );
rtx_fifo_okBytes  XRtfGet( const rtx_fifo_id        fifoId,    /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                           rtx_fifo_dataPtr         dataPtr,   /* data pointer */
                           const rtx_fifo_dataSize  dataSize,  /* data size - bytes */
                           rtx_fifo_devhandlePtr    fifoDevPtr /* rtf fifo device pointer */ );

void*  XMAlloc( const size_t  size /* size - bytes */ );
void  XMFree( void* ptr /* allocated memory pointer */ );


#ifndef __KERNEL__
RTX_RESULT  runThread( rtx_task_handlePtr        *hndPtrP,       /* handle pointer to pointer */
                       const rtx_priority         priority,      /* task priority */
                       const rtx_task_stacksize   stack,         /* stack size - bytes */
                       const rtx_task_mode        mode,          /* task mode */
                       const rtx_periodic_tick    nanoTick,      /* tick for periodic mode - nano sec */
                       rtx_periodic_tickPtr       tickPtr        /* tick space pointer */ )
/** Start RT thread in user space. **/
{
 struct sched_param  p;
 unsigned long       mytask_name;


    pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL); /* Cancel immediate on Rq
                                                                 - because parent/child sibling are not implemented */

    /* Set RT scheduling */
    switch ( priority ) {
        case RTX_TPRIOR_PERIODIC_NORMAL :
            p.sched_priority = RTX_MIN_PRIORITY_PTHREAD + 16;
            break;

        case RTX_TPRIOR_ONESHOT_NORMAL :
            p.sched_priority = RTX_MIN_PRIORITY_PTHREAD + 8;
            break;

        default :
            p.sched_priority = RTX_MAX_PRIORITY_PTHREAD;
            break;
    } /* switch */

	if ( pthread_setschedparam( pthread_self(), SCHED_FIFO, &p ) ) {
	    printx_(" -- Cannot set scheduling for RT Thread\n");
	    return RTX_FAILURE;
    }

    rt_linux_use_fpu(BCK_USE_FPU);

    mytask_name = nam2num("LXRTThread") + (unsigned long)hndPtrP;
 	if ( !(*hndPtrP = rt_task_init(mytask_name, priority, stack, 0)) ) {
		printx_(" -- Cannot Init RT Thread\n");
		return RTX_FAILURE;
	}
	printx__(" + RT Thread Init: address=%p.\n", *hndPtrP);


    mlockall(MCL_CURRENT | MCL_FUTURE); /* Can not use memory paging with RT */


    if ( mode == RTX_TASK_PERIODIC ) {
    /* Run periodic task */
	    if ( nanoTick == 0  ||  rt_task_make_periodic(*hndPtrP, rt_get_time() + nano2count(PeriodicTickBase), nano2count(nanoTick)) ) {
		    printx_(" -- Cannot run Periodic RT Thread\n");
		    return RTX_FAILURE;
        }
        *tickPtr = nanoTick;  /* Not always exactly the same that real calculated RT period value, but we need some tick info */
	}
    else {
    /* Run oneshot task */
	    if ( rt_task_resume(*hndPtrP) ) {
		    printx_(" -- Cannot run OneShot RT Thread\n");
		    return RTX_FAILURE;
        }
        *tickPtr = 0;
    }

   return RTX_SUCCESS;
}

RTX_RESULT  killThread( rtx_thread_handle  hnd /* handle */ )
/** Stop RT thread in user space. **/
{
    if ( pthread_cancel(hnd) ) {
        return RTX_FAILURE;
    }
    if ( --no_rtx_tasks < 0 ) no_rtx_tasks = 0;
   
   return RTX_SUCCESS;
}

RTX_RESULT  getMBX( rtx_mbx_handlePtr     *mbxChPtrP, /* channel pointer to pointer */
                    const rtx_mbx_msgLen   msgLen,    /* message length - bytes */
                    const rtx_mbx_chMsgs   chMsgs     /* max no messages in the channel */ )
/** Get free MBX channel in user space. **/
{
 unsigned long       mymbx_name;

    mymbx_name = nam2num("RTXMBX") + (unsigned long)mbxChPtrP;
    if ( !(*mbxChPtrP = rt_mbx_init( mymbx_name, msgLen * chMsgs )) ) {
        printx_(" -- Cannot get free MBX\n");
        return RTX_FAILURE;
    }


   return  RTX_SUCCESS;
}

RTX_RESULT  rtfCreateHnd( const rtx_fifo_id              fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                          rtx_fifo_handler_lxrt_thread   fifoHandler,/* LXRT thread - must be eternity loop and must poll fifo */
						  rtx_thread_handlePtr           thread_hndPtr/* thread handle pointer */ )
/** Run RTF FIFO fake handler **/
{
 pthread_attr_t  attr;


    if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Create Handler Error - WD FIFO Usage\n", fifoId);
	    return  RTX_FAILURE;
	}

	if ( pthread_attr_init( &attr )
         ||
         pthread_create(thread_hndPtr, &attr, fifoHandler, NULL) ) {
        printx__(" -- RTF FIFO %d Create Handler Error\n", fifoId);
        return  RTX_FAILURE;
    }


   return  RTX_SUCCESS;
}


#else
RTX_RESULT  getMBX( rtx_mbx_handlePtr      mbxChPtr, /* channel pointer */
                    const rtx_mbx_msgLen   msgLen,   /* message length - bytes */
                    const rtx_mbx_chMsgs   chMsgs    /* max no messages in the channel */ )
/** Get free MBX channel. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( rt_mbx_init( mbxChPtr, msgLen * chMsgs ) ) {
        printx_(" -- Cannot get free MBX\n");
        return RTX_FAILURE;
    }
#endif

   return  RTX_SUCCESS;
}

RTX_RESULT  rtfCreateHnd( const rtx_fifo_id  fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
#ifndef OS_WIN32
                          rtx_fifo_handler   fifoHandler /* whenever Linux process (from user space) accesses fifo - R/W,
                                                            handler routine is called - works only in kernel space */ 
#else
						  rtx_fifo_handler_win32_thread   fifoHandler, /* Win32 thread - must be eternity loop and must poll fifo */
						  rtx_thread_handlePtr            thread_hndPtr/* fake thread handle pointer */
#endif
				        )
/** Create RTF FIFO handler **/
{
    if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Create Handler Error - WD FIFO Usage\n", fifoId);
	    return  RTX_FAILURE;
	}

#ifndef OS_WIN32
    if ( rtf_create_handler( fifoId, fifoHandler ) ) {
        printx__(" -- RTF FIFO %d Create Handler Error\n", fifoId);
        return  RTX_FAILURE;
    }
#else
	*thread_hndPtr = 1;
	if ( (sRtFifoHandle[fifoId] = os_CreateTask( fifoHandler, 0, NULL, 0 )) == 0 ) {
	    *thread_hndPtr = 0;
        printx__(" -- RTF FIFO %d Create Handler Error\n", fifoId);
        return  RTX_FAILURE;
    }
#endif

   return  RTX_SUCCESS;
}


#endif  /* __KERNEL__ */



RTX_RESULT  XRunTask( rtx_task_handlePtr        hndPtr,       /* handle pointer */
                      const rtx_priority        priority,     /* task priority */
                      const rtx_task_stacksize  stack,        /* stack size - bytes */
                      const rtx_task_mode       mode,         /* task mode */
                      const rtx_periodic_tick   nanoTick,     /* tick for periodic mode - nano sec */
                      rtai_task                 task_entry,   /* entry point */
                      rtx_thread_handlePtr      thread_hndPtr,/* thread handle pointer */
					  const rtx_task_arg        arg           /* argument */ )
/** Start RT task in kernel space. **/
{
#ifdef OS_WIN32
	if ( ++no_rtx_tasks > RTX_MAX_TASKS_NO ) {
        --no_rtx_tasks;
		printx_(" -- Cannot Run RT Task - Over max tasks\n");
        return RTX_FAILURE;
	}

	hndPtr =(void **) os_CreateTask(task_entry, (void *)arg, NULL, stack);
#else

#ifdef __KERNEL__
 int  res;

	if (thread_hndPtr) {}

	if ( ++no_rtx_tasks > RTX_MAX_TASKS_NO ) {
        --no_rtx_tasks;
		printx_(" -- Cannot Run RT Task - Over max tasks\n");
        return RTX_FAILURE;
	}

	if ( (res = rt_task_init(hndPtr, task_entry, arg, stack, priority, 0, 0)) != 0 ) {
        --no_rtx_tasks;
        printx_(" -- Cannot Init RT Task (address=%p), errno=%d\n", hndPtr, -res);
        return RTX_FAILURE;
    }
    printx__(" + RT Task Init: address=%p.\n", hndPtr);

    if ( mode == RTX_TASK_PERIODIC ) {
    /* Run periodic task */
        if ( rt_task_make_periodic(hndPtr, rt_get_time() + (nano2count(PeriodicTickBase) * no_rtx_tasks), nano2count(nanoTick)) ) {
            printx_(" -- Cannot run Periodic RT Task\n");
            return RTX_FAILURE;
        }
    }
    else {
    /* Run oneshot task */
	    if ( rt_task_resume(hndPtr) ) {
		    printx_(" -- Cannot run OneShot RT Task\n");
		    return RTX_FAILURE;
        }
    }
#else
 pthread_attr_t  attr;

    if (hndPtr && priority && stack && mode && nanoTick) {}

	if ( ++no_rtx_tasks > RTX_MAX_TASKS_NO ) {
        --no_rtx_tasks;
		printx_(" -- Cannot Run RT Task - Over max tasks\n");
        return RTX_FAILURE;
	}

	if ( pthread_attr_init( &attr ) ) {
        --no_rtx_tasks;
        printx_(" -- RT Thread attr Error\n");
        return RTX_FAILURE;
    }
	if ( pthread_create(thread_hndPtr, &attr, task_entry, (void*)&arg) ) {
        --no_rtx_tasks;
        printx_(" -- RT Thread Create Error\n");
        return RTX_FAILURE;
    }
#endif
#endif


   return RTX_SUCCESS;
}


RTX_RESULT  XKillTask( rtx_task_handlePtr  hndPtr,   /* handle pointer */
                       rtx_thread_handle   hndThread /* thread handle */ )
/** Stop RT thread. **/
{
#ifdef OS_WIN32
   os_KillTask(hndPtr, 1);
   if ( --no_rtx_tasks < 0 ) no_rtx_tasks = 0;
#else

#ifdef __KERNEL__
    if ( !hndPtr )  return RTX_FAILURE;

    if ( --no_rtx_tasks < 0 ) no_rtx_tasks = 0;

    if ( rt_task_delete(hndPtr) ) {
       /* printx_(" -- Cannot delete RT Task\n"); don't print message - task is probably already dead */
        return RTX_FAILURE;
    }
#else
    if ( !hndThread ) {
        printx_(" -- Cannot delete RT Task - NULL Thread handle\n");
        return RTX_FAILURE;
    }
	else
        Xkill( hndThread );
#endif
#endif

   return RTX_SUCCESS;
}


void  XExit( void )
{
#ifdef OS_WIN32
    os_ExitTask(1);
#else

#ifndef __KERNEL__
    Xkill(pthread_self());
#else
    Xkill(rt_whoami(),0);
#endif
#endif

   return;
}


rtx_task_handlePtr  XGetId( void )
/** Get current task id. **/
{
#ifdef OS_WIN32
   return (rtx_task_handlePtr )os_GetTaskID();
#else  

#ifndef __KERNEL__
    return  rt_buddy();
#else    
    return  rt_whoami();
#endif    
#endif
}


void  XLock( void )
/** Lock RT scheduler. **/
{
#ifdef OS_WIN32
  return;
#else
#ifndef __KERNEL__
    printx_(" -- Cannot lock scheduler for RT Thread\n");

   return;
#else
    return  rt_sched_lock();
#endif    
#endif
}
                                                                                                                           
void  XUnLock( void )
/** Unlock RT scheduler. **/
{
#ifdef OS_WIN32
  return;
#else
#ifndef __KERNEL__
    printx_(" -- Cannot unlock scheduler for RT Thread\n");

   return;
#else
    return  rt_sched_unlock();
#endif
#endif
}

                                                                                                                           
RTX_RESULT  XCPri( rtx_task_handlePtr  hndPtr,  /* handle pointer */
                   const rtx_priority  priority /* priority */ )
/** Change priority. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( !hndPtr )  return RTX_FAILURE;

    if ( rt_change_prio( hndPtr, priority ) < 0 ) {
	    printx_(" -- Cannot change priority for RT Task\n");
        return RTX_FAILURE;
    }
#endif


   return RTX_SUCCESS;
}


void  XDelay( rtx_periodic_tickno         ticks,  /* no ticks */
              const rtx_periodic_tickPtr  tickPtr /* tick pointer */ )
/** Sleep no of ticks for periodic task. **/
{
#ifdef OS_WIN32
   os_Sleep(ticks);
#else
   int  i;

    if ( (int)XGetTickInterv( tickPtr ) == 0 )
    /* OneShot task */
        return;

    for ( i = 0; i < ticks; ++i )
		rt_task_wait_period();
#endif

   return;
}


void  XDelayShort( rtx_nano_time  ntime /* no nano secs */ )
/** Sleep no of nano seconds. **/
{
#ifdef OS_WIN32
   os_Sleep((ULONG )(ntime / 1000000));
#else
   rt_sleep(nano2count(ntime));
#endif

   return;
}


RTX_RESULT  XStop( rtx_task_handlePtr  hndPtr /* handle pointer */ )
/** Suspend thread/task. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( !hndPtr )  return RTX_FAILURE;

    if ( rt_task_suspend( hndPtr ) < 0 ) {
	    printx_(" -- Cannot suspend RT task\n");
        return RTX_FAILURE;
    }
#endif

   return RTX_SUCCESS;     
}


RTX_RESULT  XResume( rtx_task_handlePtr  hndPtr /* handle pointer */ )
/** Resume thread/task. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( !hndPtr )  return RTX_FAILURE;

    if ( rt_task_resume( hndPtr ) < 0 ) {
	    printx_(" -- Cannot resume RT task\n");
        return RTX_FAILURE;
    }
#endif

   return RTX_SUCCESS;
}


rtx_periodic_tick  XGetTickInterv( const rtx_periodic_tickPtr  tickPtr /* tick pointer */ )
/** Get nano seconds tick duration. **/
{
#ifdef OS_WIN32
   return *tickPtr;
#else

#ifndef __KERNEL__
   return  *tickPtr;  /* Not always exactly the same that real RT period value, but that's all we have */
#else
   if (tickPtr) {}

   return count2nano(rt_whoami()->period);
#endif
#endif
}


RTX_RESULT  XSetTickInterv( rtx_periodic_tickPtr  tickPtr,     /* tick pointer */
                            const rtx_periodic_tick  nanoTick  /* tick for periodic mode - nano sec */ )
/** Set nano seconds tick duration for periodic thread/task. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
#ifndef __KERNEL__
    if ( *tickPtr == 0 ) {
        printx_(" -- Cannot set tick for OneShot RT thread\n");
		return RTX_FAILURE;
    }

    if ( rt_set_period(rt_buddy(), nano2count(nanoTick)) < 0 ) {
        printx_(" -- Cannot set tick for RT thread\n");
		return RTX_FAILURE;
    }
    *tickPtr = nanoTick;  /* Not always exactly the same that real calculated RT period value, but we need some tick info */
#else
    if ( rt_whoami()->period == 0 ) {
        printx_(" -- Cannot set tick for OneShot RT task\n");
		return RTX_FAILURE;
    }

    if ( rt_set_period(rt_whoami(), nano2count(nanoTick)) < 0 ) {
        printx_(" -- Cannot set tick for RT task\n");
		return RTX_FAILURE;
    }
#endif
#endif

   return  RTX_SUCCESS;
}


rtx_periodic_tickno  XGetNoTicksTrunc( const rtx_nano_time         ntime,  /* no nano secs */
                                       const rtx_periodic_tickPtr  tickPtr /* tick pointer */ )
/** Get truncated no ticks from nano seconds for periodic thread/task. **/
{
 int  tick = (int)XGetTickInterv( tickPtr );

    if ( tick == 0 )
    /* OneShot task */
        return 0;


   return (rtx_periodic_tickno)((int)ntime / tick);
}


rtx_periodic_tickno  XGetNoTicksRoundUp( const rtx_nano_time         ntime,  /* no nano secs */
                                         const rtx_periodic_tickPtr  tickPtr /* tick pointer */ )
/** Get roundedup no ticks from nano seconds for periodic thread/task. **/
{
 int  tick = (int)XGetTickInterv( tickPtr );

    if ( tick == 0 )
    /* OneShot task */
        return 0;


   return ((int)ntime % tick ? XGetNoTicksTrunc(ntime, tickPtr) + 1 : XGetNoTicksTrunc(ntime, tickPtr));
}


rtx_periodic_tickno  XGetNoTicks( const rtx_nano_time         ntime,  /* no nano secs */
                                  const rtx_periodic_tickPtr  tickPtr /* tick pointer */ )
/** Get no needed ticks from nano seconds for periodic thread/task. **/
{
   return  XGetNoTicksRoundUp( ntime, tickPtr );
}


RTX_RESULT  XReleaseMBX( rtx_mbx_handlePtr     mbxChPtr /* channel pointer */ )
/** Release MBX channel. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( !mbxChPtr )  return RTX_FAILURE;

#ifndef __KERNEL__
    if ( !rt_mbx_delete( mbxChPtr ) ) {
#else
    if ( rt_mbx_delete( mbxChPtr ) ) {
#endif
        printx_(" -- Cannot delete MBX\n");
        return RTX_FAILURE;
    }
#endif


   return  RTX_SUCCESS;
}


RTX_RESULT  XSendMBX( rtx_mbx_handlePtr         mbxChPtr,      /* channel pointer */
                      const rtx_mbx_msgPtr      mbxMsgPtr,     /* message pointer */
                      const rtx_mbx_msgLen      msgLen,        /* message length - bytes */
                      const rtx_nano_time       busyWait,      /* timeout if channel is bussy - nano secs - 0 don't wait */
                      const rtx_nano_time       readTaskWait,  /* timeout (after busyWait) to be read and deleted by task - nano secs - 0 don't wait */
                      const rtx_task_handlePtr  readTaskHndPtr /* read task handle pointer */ )
/** Send MBX message. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
 unsigned int  rpcMsg = 0;
 
    if ( !mbxChPtr )  return RTX_FAILURE;

    if ( rt_mbx_send_timed( mbxChPtr, mbxMsgPtr, msgLen, nano2count(busyWait)) ) {
        printx_(" -- Cannot send MBX message\n");
        return RTX_FAILURE;
    }

    if ( readTaskWait ) { /* Wait the answer that message was read by task */
        if ( rt_rpc_timed( readTaskHndPtr, rpcMsg, &rpcMsg, nano2count(readTaskWait)) != readTaskHndPtr ) {
            printx_(" -- MBX message RPC request read Error\n");
            return RTX_FAILURE;
        }
    }
#endif

   return  RTX_SUCCESS;
}                             


RTX_RESULT  XWaitMBX( rtx_mbx_handlePtr     mbxChPtr,  /* channel pointer */
                      rtx_mbx_msgPtr        mbxMsgPtr, /* message pointer */
                      const rtx_mbx_msgLen  msgLen,    /* message length - bytes */
                      const rtx_nano_time   arriveWait /* timeout if channel is empty - nano secs - 0 don't wait */ )
/** Wait MBX message - get it and put it back into the channel - FIFO order. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( !mbxChPtr )  return RTX_FAILURE;

  /**
   * Must ensure sync/lock :
   * rt_mbx_send_timed after rt_mbx_receive_timed must be called before any other send to this channel
   **/

    if ( rt_mbx_receive_timed( mbxChPtr, mbxMsgPtr, msgLen, nano2count(arriveWait)) ) {
        printx_(" -- Wait MBX message Error\n");
        return RTX_FAILURE;
    }

    if ( rt_mbx_send_timed( mbxChPtr, mbxMsgPtr, msgLen, 0) ) {
        printx_(" -- Cannot reSend MBX (FIFO) message\n");
        return RTX_FAILURE;
    }
#endif
   
   return  RTX_SUCCESS;
}


RTX_RESULT  XReadMBX( rtx_mbx_handlePtr         mbxChPtr,       /* channel pointer */
                      rtx_mbx_msgPtr            mbxMsgPtr,      /* message pointer */
                      const rtx_mbx_msgLen      msgLen,         /* message length - bytes */
                      const rtx_nano_time       arriveWait,     /* timeout if channel is empty - nano secs - 0 don't wait */
                      const rtx_task_handlePtr  replyTaskHndPtr /* reply task handle pointer - NULL don't reply */ )
/** Read MBX message - delete message from the channel. **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
 unsigned int  rpcMsg = 0;
 
    if ( !mbxChPtr )  return RTX_FAILURE;

    if ( rt_mbx_receive_timed( mbxChPtr, mbxMsgPtr, msgLen, nano2count(arriveWait)) ) {
        printx_(" -- Read MBX message Error\n");
        return RTX_FAILURE;
    }

    if ( replyTaskHndPtr ) { /* Reply to the sender task */
        if ( rt_receive_timed( replyTaskHndPtr, &rpcMsg, 0) != replyTaskHndPtr
             ||
             rt_return( replyTaskHndPtr, rpcMsg ) != replyTaskHndPtr ) {
            printx_(" -- MBX message RPC reply read Error\n");
            return RTX_FAILURE;
        }
    }
#endif

   return  RTX_SUCCESS;
}                             


RTX_RESULT  XGetFreeMuteX( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ )
/** Get free mutex **/
{
#ifdef OS_WIN32
    if ( 0 == ( *mxPtr = (unsigned int)CreateMutex(NULL, FALSE, NULL ) ) ) {
#else
    if ( pthread_mutex_init( mxPtr, NULL ) ) {
#endif
        printx_(" -- Mutex Init Error\n");
        return  RTX_FAILURE;
    }

   return  RTX_SUCCESS;
}


RTX_RESULT  XReleaseMuteX( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ )
/** Release mutex **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
    if ( pthread_mutex_destroy( mxPtr ) ) {
        printx_(" -- Mutex Destroy Error\n");
        return  RTX_FAILURE;
    }
#endif

   return  RTX_SUCCESS;
}


RTX_RESULT  XLockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ )
/** Lock section - blocked wait if locked **/
{
#ifdef OS_WIN32
    if ( WaitForSingleObject( (HANDLE)*mxPtr, INFINITE) != WAIT_OBJECT_0 ) {
//    if ( WaitForSingleObject( *mxPtr, 15000L) != WAIT_OBJECT_0 ) {
#else
    if ( pthread_mutex_lock( mxPtr ) ) {
#endif
        printx_(" -- Lock Section Error\n");
        return  RTX_FAILURE;
    }

   return  RTX_SUCCESS;
}


RTX_RESULT  XUnlockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ )
/** Unlock section - always unlock **/
{
#ifdef OS_WIN32
    if ( !ReleaseMutex((HANDLE)*mxPtr) )
#else
    if ( pthread_mutex_unlock( mxPtr ) )
#endif
        return  RTX_FAILURE;

   return  RTX_SUCCESS;
}


RTX_RESULT  XTryLockSection( rtx_mutex_handlePtr   mxPtr /* mutex pointer */ )
/** Try lock section - lock if unlocked, else return failure **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else
 int  res;

    if ( (res = pthread_mutex_trylock( mxPtr )) == EBUSY ) {
        return  RTX_FAILURE;  /* section is locked */
    }

    if ( res ) {
        printx_(" -- TryLock Section Error\n");
        return  RTX_FAILURE;
    }
#endif

   return  RTX_SUCCESS;
}


RTX_RESULT  XRtfCreate( const rtx_fifo_id     fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                        const rtx_fifo_size   fifoSize,   /* rtf fifo size - bytes */
                        rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                        rtx_fifo_sizePtr      fifoSizePtr /* rtf fifo size space pointer */ )
/** Create RTF FIFO **/
{
#ifdef OS_WIN32
   // Nothing todo here
#else
#ifndef __KERNEL__
 char devName[12] = "/dev/rtf\0\0\0\0";


	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Create Error - WD FIFO Usage\n", fifoId);
	    return  RTX_FAILURE;
	}

    if ( fifoId < 10 )
        devName[8] = (char)((int)'0' + fifoId);
    else {
        devName[8] = (char)((int)'0' + fifoId/10);
        devName[9] = (char)((int)'0' + fifoId%10);
    }

	if ( *fifoDevPtr != INVALID_RTF_HND )
        close(*fifoDevPtr);
    if ( (*fifoDevPtr = rtf_open_sized(devName, O_RDWR, fifoSize)) < 0 ) {
        printx__(" -- RTF FIFO %d Create Error\n", fifoId);
        return  RTX_FAILURE;
    }
    *fifoSizePtr = fifoSize;

#else
    if (fifoDevPtr) {}

	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Create Error - WD FIFO Usage\n", fifoId);
	    return  RTX_FAILURE;
	}
    
	rtf_reset( fifoId );
    if ( rtf_create( fifoId, fifoSize ) ) {
        printx__(" -- RTF FIFO %d Create Error\n", fifoId);
        return  RTX_FAILURE;
    }
    *fifoSizePtr = fifoSize;
#endif
#endif

   return RTX_SUCCESS;
}


void  XRtfDestroy( const rtx_fifo_id     fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                   rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                   rtx_fifo_sizePtr      fifoSizePtr,/* rtf fifo size space pointer */
				   rtx_thread_handlePtr  thread_hndPtr/* thread handle pointer */ )
/** Destroy RTF FIFO **/
{
#ifdef OS_WIN32
	*thread_hndPtr = 0; /* exit fifo handler thread */
	os_Sleep(200);      /* wait fifo handler to exit */
	os_CloseEvent(sRtFifoHandle[fifoId]); /* close handler */
#else

#ifndef __KERNEL__
	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Destroy Error - WD FIFO Usage\n", fifoId);
	    return;
	}

	if ( *fifoDevPtr != INVALID_RTF_HND ) {
        if ( thread_hndPtr )
		    Xkill( *thread_hndPtr );
		close(*fifoDevPtr);
        *fifoDevPtr = INVALID_RTF_HND;
        *fifoSizePtr = 0;
    }

#else
    if (fifoDevPtr) {}

	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Destroy Error - WD FIFO Usage\n", fifoId);
	    return;
	}
	
	rtf_destroy( fifoId );
    *fifoSizePtr = 0;
#endif
#endif

   return;
}


RTX_RESULT  XRtfReset( const rtx_fifo_id     fifoId,     /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                       rtx_fifo_devhandlePtr fifoDevPtr, /* rtf fifo device pointer */
                       rtx_fifo_sizePtr      fifoSizePtr /* rtf fifo size space pointer */ )
/** Reset RTF FIFO - discard all data **/
{
#ifdef OS_WIN32
   // TODO: implementation
#else

#ifndef __KERNEL__

	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Reset Error - WD FIFO Usage\n", fifoId);
	    return  RTX_FAILURE;
	}

	XRtfCreate( fifoId, *fifoSizePtr, fifoDevPtr, fifoSizePtr );

#else
    if (fifoDevPtr && fifoSizePtr) {}

	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Reset Error - WD FIFO Usage\n", fifoId);
	    return  RTX_FAILURE;
	}
    
	if ( rtf_reset( fifoId ) ) {
        printx__(" -- RTF FIFO %d Reset Error\n", fifoId);
        return  RTX_FAILURE;
    }
#endif
#endif

   return  RTX_SUCCESS;
}


rtx_fifo_okBytes  XRtfPut( const rtx_fifo_id        fifoId,   /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                           const rtx_fifo_dataPtr   dataPtr,  /* data pointer */
                           const rtx_fifo_dataSize  dataSize, /* data size - bytes */
                           rtx_fifo_devhandlePtr    fifoDevPtr /* rtf fifo device pointer */ )
/** Put RTF FIFO - write data into the fifo within RT task **/
{
#ifdef OS_WIN32
  int BytesWritten = 0;

  FILE *sRtFifoHndLoc;

  if( XLockSection( (unsigned int *)&hMutex ) != RTX_SUCCESS)
    return -1;

  sRtFifoHnd[fifoId] = sRtFifoHndLoc = fopen(sRtFifoNames[fifoId], "a+b");
  if (sRtFifoHndLoc)
  {
    BytesWritten = fwrite(dataPtr, 1, dataSize, sRtFifoHndLoc);
    fclose(sRtFifoHndLoc);
  }

  if ( XUnlockSection( (unsigned int *)&hMutex ) != RTX_SUCCESS)
    return 0;
 
  return BytesWritten;
#else

#ifndef __KERNEL__
	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Put Error - WD FIFO Usage\n", fifoId);
	    return  0;
	}

   return  rtf_write_timed( *fifoDevPtr, dataPtr, dataSize, 1 /*1 ms*/);

#else
    if (fifoDevPtr) {}

	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Put Error - WD FIFO Usage\n", fifoId);
	    return  0;
	}

   return  rtf_put( fifoId, dataPtr, dataSize );
#endif
#endif
}


rtx_fifo_okBytes  XRtfGet( const rtx_fifo_id        fifoId,    /* rtf fifo id - 2,...,63  for /dev/rtf2,...,/dev/rtf63 devices */
                           rtx_fifo_dataPtr         dataPtr,   /* data pointer */
                           const rtx_fifo_dataSize  dataSize,  /* data size - bytes */
                           rtx_fifo_devhandlePtr    fifoDevPtr /* rtf fifo device pointer */ )
/** Get RTF FIFO - read data from the fifo within RT task **/
{
#ifdef OS_WIN32
  int BytesRead;
  long CurrPos;

  volatile rtx_fifo_id fifoIdLoc = fifoId;
  FILE *sRtFifoHndLoc;

  if( XLockSection( (unsigned int *)&hMutex ) != RTX_SUCCESS)
    return -1;

  sRtFifoHnd[fifoIdLoc] = sRtFifoHndLoc = fopen(sRtFifoNames[fifoIdLoc], "rb");
  if (!sRtFifoHndLoc)
  {
    XUnlockSection( (unsigned int *)&hMutex );
    return 0;
  }
                
  fseek(sRtFifoHndLoc, 0, SEEK_END);
  CurrPos = ftell(sRtFifoHndLoc);

  if (CurrPos < sRtFifoSeek[fifoIdLoc])
  {
    sRtFifoSeek[fifoIdLoc] = CurrPos;
  }

  fseek(sRtFifoHndLoc, sRtFifoSeek[fifoIdLoc], SEEK_SET);
  BytesRead = fread(dataPtr, 1, dataSize, sRtFifoHndLoc);

  sRtFifoSeek[fifoIdLoc] += BytesRead;
  fclose(sRtFifoHndLoc);

  if ( XUnlockSection( (unsigned int *)&hMutex ) != RTX_SUCCESS)
    return 0;

  return BytesRead;
#else

#ifndef __KERNEL__
	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Get Error - WD FIFO Usage\n", fifoId);
	    return  0;
	}

   return  rtf_read_timed( *fifoDevPtr, dataPtr, dataSize, 1 /*1 ms*/);

#else
    if (fifoDevPtr) {}

	if ( fifoId == WD_CMD_FIFO  ||  fifoId == WD_DATA_FIFO ) {
        printx__(" -- RTF FIFO %d Get Error - WD FIFO Usage\n", fifoId);
	    return  0;
	}

   return  rtf_get( fifoId, dataPtr, dataSize );
#endif
#endif
}


void*  XMAlloc( const size_t  size /* size - bytes */ )
/** Alloc space. **/
{
#ifdef OS_WIN32
   return os_MemAlloc(size);
#else

#ifndef __KERNEL__
    return  malloc(size);
#else
    return  kmalloc(size, GFP_KERNEL);
#endif
#endif
}


void  XMFree( void* ptr /* allocated memory pointer */ )
/** Free space. **/
{
#ifdef OS_WIN32
   os_MemFree(ptr);
#else

#ifndef __KERNEL__
    free(ptr);
    ptr = NULL;
#else
    kfree(ptr);
    ptr = NULL;
#endif
#endif

   return;
}

/* #########################################################################
  ! END !  RTX  Functions
   ######################################################################### */



/* #########################################################################
    Hard RT  RTX-RTAI  in kernel space
   ######################################################################### */
#ifdef __KERNEL__

int init_module( void )
{
#ifdef OS_WIN32
  int fifoId;
  
  FILE *sRtFifoHndLoc;


if ( NULL == (hMutex = CreateMutex(NULL, FALSE, NULL ) ) ) {
//if ( XGetFreeMuteX( &FIFOmutex ) == RTX_FAILURE ) {
	return -1;
}
//InitializeCriticalSection(&criticalSection);

   _mkdir(OS_RTF_FIFO_DIR);

   for (fifoId = 0; fifoId < MAX_FIFOS; fifoId++)
   {
     sRtFifoSeek[fifoId] = 0;
     sprintf(sRtFifoNames[fifoId], OS_RTF_FIFO_DIR"rtf%d", fifoId);

     if( XLockSection( (unsigned int *)&hMutex ) != RTX_SUCCESS)
       return -1;

     sRtFifoHnd[fifoId] = sRtFifoHndLoc = fopen(sRtFifoNames[fifoId], "wb");
     if (sRtFifoHndLoc)
     {
       fclose(sRtFifoHndLoc);
     }

     if ( XUnlockSection( (unsigned int *)&hMutex ) != RTX_SUCCESS)
       return -1;

   }
#else

   stop_rt_timer();  /* stop current timer */

	rt_linux_use_fpu(BCK_USE_FPU); /* we don't use FPU on target machine */

#if 0
	/* We use OneShot global timer */
	rt_set_oneshot_mode();
    rt_preempt_always(1);  /* interruptable by higher priorities */

	start_rt_timer(nano2count(0));

	printk("*** RTX: Start RTAI oneshot timer preemptive\n");
#else
	/* We use Periodic global timer */
    rt_set_periodic_mode();

	start_rt_timer((int)nano2count(PeriodicTickBase));

#ifdef __arm__
    rtai_hz = 1000000000 / PeriodicTickBase;
    rtai_tick = (1000000 + rtai_hz/2) / rtai_hz;
    rtai_latch = (CLOCK_TICK_RATE + rtai_hz/2) / rtai_hz;
    rtai_tick_cnt = 0;
#endif

    printk("*** RTX: Start RTAI periodic timer with period %d ns\n", PeriodicTickBase);
#endif
#endif

   return 0;
}


void cleanup_module( void )
{
#ifdef OS_WIN32
   // Nothing todo here
#else
    /* Stop the timer */
	stop_rt_timer();

#ifdef __arm__
    rtai_hz = HZ;
    rtai_tick = (1000000 + rtai_hz/2) / rtai_hz;
    rtai_latch = (CLOCK_TICK_RATE + rtai_hz/2) / rtai_hz;
    rtai_tick_cnt = 0;
#endif
#endif

   return;
}


#ifdef OS_WIN32
int Rtx_Init( void )
{
	return init_module();
}

void Rtx_Destroy( void )
{
	cleanup_module();
}

MODULE_INIT(Rtx_Init);
MODULE_FREE(Rtx_Destroy);
#endif


/* Export */
EXPORT_SYMBOL(XRunTask);
EXPORT_SYMBOL(XKillTask);
EXPORT_SYMBOL(XExit);
EXPORT_SYMBOL(XGetId);
EXPORT_SYMBOL(XLock);
EXPORT_SYMBOL(XUnLock);
EXPORT_SYMBOL(XCPri);
EXPORT_SYMBOL(XDelay);
EXPORT_SYMBOL(XDelayShort);
EXPORT_SYMBOL(XStop);
EXPORT_SYMBOL(XResume);
EXPORT_SYMBOL(XGetTickInterv);
EXPORT_SYMBOL(XSetTickInterv);
EXPORT_SYMBOL(XGetNoTicksTrunc);
EXPORT_SYMBOL(XGetNoTicksRoundUp);
EXPORT_SYMBOL(XGetNoTicks);
EXPORT_SYMBOL(XReleaseMBX);
EXPORT_SYMBOL(XSendMBX);
EXPORT_SYMBOL(XWaitMBX);
EXPORT_SYMBOL(XReadMBX);
EXPORT_SYMBOL(XGetFreeMuteX);
EXPORT_SYMBOL(XReleaseMuteX);
EXPORT_SYMBOL(XLockSection);
EXPORT_SYMBOL(XUnlockSection);
EXPORT_SYMBOL(XTryLockSection);
EXPORT_SYMBOL(XRtfCreate);
EXPORT_SYMBOL(XRtfDestroy);
EXPORT_SYMBOL(XRtfReset);
EXPORT_SYMBOL(XRtfPut);
EXPORT_SYMBOL(XRtfGet);
EXPORT_SYMBOL(XMAlloc);
EXPORT_SYMBOL(XMFree);
EXPORT_SYMBOL(getMBX);
EXPORT_SYMBOL(rtfCreateHnd);

#endif  /* __KERNEL__ */

/* #########################################################################
  ! THE  END !
   ######################################################################### */



